<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;

class VendaController extends Controller
{
    /* ============================
        LISTAGEM DE VENDAS
    ============================ */
    public function index()
    {
        $vendas = DB::table('vendas')
            ->join('usuarios', 'usuarios.codigo_us', '=', 'vendas.usuario')
            ->select(
                'vendas.codigo',
                'usuarios.nome_us as usuario',
                'vendas.nr_itens',
                'vendas.valorPago',
                'vendas.data',
                'vendas.metodo_pagamento',
                'vendas.valor_recebido',
                'vendas.troco',
                'vendas.referencia'
            )
            ->orderBy('vendas.data', 'desc')
            ->get();

        return view('vendas.index', compact('vendas'));
    }

    /* ============================
        DETALHES PARA MODAL
    ============================ */
    public function detalhes($codigo)
    {
        $venda = DB::table('vendas')
            ->join('usuarios', 'usuarios.codigo_us', '=', 'vendas.usuario')
            ->select(
                'vendas.codigo',
                'usuarios.nome_us as usuario',
                'vendas.data',
                'vendas.valorPago',
                'vendas.metodo_pagamento',
                'vendas.valor_recebido',
                'vendas.troco',
                'vendas.referencia'
            )
            ->where('vendas.codigo', $codigo)
            ->first();

        $itens = DB::table('itensvenda')
            ->join('produtos', 'itensvenda.codProduto', '=', 'produtos.codigo_pro')
            ->select(
                'produtos.nome_pro',
                'itensvenda.preco',
                'itensvenda.quantidade',
                'itensvenda.subtotal'
            )
            ->where('codigoVenda', $codigo)
            ->get();

        return response()->json([
            'venda' => $venda,
            'itens' => $itens
        ]);
    }

    /* ============================
        FINALIZAR VENDA (CAIXA)
    ============================ */
    public function finalizarVenda(Request $request)

    {
        
        DB::beginTransaction();

        try {
            // Usuário logado
            $usuarioLogado = auth()->user();
            if (!$usuarioLogado) {
                return response()->json([
                    'error' => 'Usuário não autenticado.'
                ], 401);
            }

            $itens = $request->itens;
            if (!$itens || count($itens) === 0) {
                return response()->json(['error' => 'Nenhum item enviado!'], 400);
            }

            // Validação de estoque antes de salvar
            foreach ($itens as $i) {
                $produto = DB::table('produtos')
                    ->where('codigo_pro', $i['codigo'])
                    ->lockForUpdate()
                    ->first();

                if (!$produto) {
                    throw new \Exception("Produto {$i['codigo']} não encontrado.");
                }

                $qtdSolicitada = (int) $i['quant'];
                $qtdDisponivel = (int) $produto->quant_pro;

                if ($qtdDisponivel < $qtdSolicitada) {
                    throw new \Exception("Estoque insuficiente para o produto {$produto->nome_pro}. Disponível: {$qtdDisponivel}, solicitado: {$qtdSolicitada}");
                }
            }

            // Calcular valores
            $total = 0.0;
            $nr_itens = 0;
            foreach ($itens as $i) {
                $nr_itens += (int)$i['quant'];
                $total += (float)$i['preco'] * (int)$i['quant'];
            }

            // Gera código da venda
            $ultimo = DB::table('vendas')->orderBy('codigo', 'desc')->first();
            $novoCodigo = $ultimo
                ? 'FACT' . str_pad(((int)substr($ultimo->codigo, 4)) + 1, 4, '0', STR_PAD_LEFT)
                : 'FACT0001';

            // Salvar venda
            DB::table('vendas')->insert([
                'codigo'           => $novoCodigo,
                'usuario'          => $usuarioLogado->codigo_us,
                'nr_itens'         => $nr_itens,
                'valorPago'        => $total,
                'data'             => now(),
                'metodo_pagamento' => $request->metodo_pagamento ?? null,
                'valor_recebido'   => $request->valor_recebido ?? null,
                'troco'            => $request->troco ?? null,
                'referencia'       => $request->referencia ?? null
            ]);

            // Salvar itens e baixar estoque
            foreach ($itens as $i) {
                DB::table('produtos')
                    ->where('codigo_pro', $i['codigo'])
                    ->decrement('quant_pro', (int)$i['quant']);

                DB::table('itensvenda')->insert([
                    'codigoVenda' => $novoCodigo,
                    'codProduto'  => $i['codigo'],
                    'preco'       => (float)$i['preco'],
                    'quantidade'  => (int)$i['quant'],
                    'subtotal'    => (float)$i['preco'] * (int)$i['quant']
                ]);
            }

            DB::commit();

            return response()->json([
                'success'     => true,
                'codigoVenda' => $novoCodigo
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'error'    => 'Erro ao finalizar venda',
                'detalhes' => $e->getMessage()
            ], 500);
        }
    }

    /* ============================
        EXPORTAR PDF
    ============================ */
    public function exportarPDF($codigo)
    {
        $venda = DB::table('vendas')
            ->join('usuarios', 'usuarios.codigo_us', '=', 'vendas.usuario')
            ->select('vendas.*', 'usuarios.nome_us as usuario')
            ->where('vendas.codigo', $codigo)
            ->first();

        $itens = DB::table('itensvenda')
            ->join('produtos', 'itensvenda.codProduto', '=', 'produtos.codigo_pro')
            ->select('produtos.nome_pro', 'itensvenda.preco', 'itensvenda.quantidade', 'itensvenda.subtotal')
            ->where('codigoVenda', $codigo)
            ->get();

        $total = $itens->sum('subtotal');

        $pdf = Pdf::loadView('vendas.pdf', compact('venda', 'itens', 'total'));

        return $pdf->download('Venda_' . $codigo . '.pdf');
    }
}
