<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Usuario;

class UsuarioController extends Controller
{
    // ---------------------------------------
    // LISTAGEM DE USUÁRIOS
    // ---------------------------------------
    public function index()
    {
        // Ordena por código
        $usuarios = Usuario::orderBy('codigo_us', 'asc')->get();

        // Obter próximo código para o modal
        $ultimo = Usuario::orderBy('codigo_us', 'desc')->first();
        if ($ultimo && preg_match('/USU(\d+)/', $ultimo->codigo_us, $matches)) {
            $numero = intval($matches[1]) + 1;
            $proximoCodigo = 'USU' . str_pad($numero, 4, '0', STR_PAD_LEFT);
        } else {
            $proximoCodigo = 'USU0001';
        }

        return view('usuarios.index', compact('usuarios', 'proximoCodigo'));
    }

    // ---------------------------------------
    // GERAR PRÓXIMO CÓDIGO (Ex: USU0001)
    // ---------------------------------------
    public function nextCode()
    {
        $ultimo = Usuario::orderBy('codigo_us', 'desc')->first();

        if ($ultimo && preg_match('/USU(\d+)/', $ultimo->codigo_us, $matches)) {
            $numero = intval($matches[1]) + 1;
            $next = 'USU' . str_pad($numero, 4, '0', STR_PAD_LEFT);
        } else {
            $next = 'USU0001';
        }

        return response()->json(['next' => $next]);
    }

    // ---------------------------------------
    // CADASTRAR NOVO USUÁRIO
    // ---------------------------------------
    public function store(Request $request)
    {
        $request->validate([
            'codigo_us' => 'required|unique:usuarios,codigo_us',
            'nome_us'   => 'required|string',
            'sexo_us'   => 'required|string',
            'tipo_us'   => 'required|string',
            'acesso_us' => 'required|string',
            'senha'     => 'required|string|min:4',
        ]);

        // Cria usuário (senha será hash automático no model)
        Usuario::create($request->all());

        return response()->json(['success' => true]);
    }

    // ---------------------------------------
    // ATUALIZAR USUÁRIO
    // ---------------------------------------
    public function update(Request $request, $codigo_us)
    {
        $usuario = Usuario::where('codigo_us', $codigo_us)->firstOrFail();

        $request->validate([
            'nome_us'   => 'required|string',
            'sexo_us'   => 'required|string',
            'tipo_us'   => 'required|string',
            'acesso_us' => 'required|string',
        ]);

        $usuario->nome_us   = $request->nome_us;
        $usuario->sexo_us   = $request->sexo_us;
        $usuario->tipo_us   = $request->tipo_us;
        $usuario->acesso_us = $request->acesso_us;

        // Atualizar senha apenas se fornecida
        if ($request->filled('senha')) {
            $usuario->senha = $request->senha; // hash automático no model
        }

        $usuario->save();

        return response()->json(['success' => true]);
    }

    // ---------------------------------------
    // APAGAR USUÁRIO
    // ---------------------------------------
    public function destroy($codigo_us)
    {
        $usuario = Usuario::where('codigo_us', $codigo_us)->firstOrFail();
        $usuario->delete();

        return response()->json(['success' => true]);
    }
}
