<?php

namespace App\Http\Controllers;

use App\Models\Produto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ProdutoController extends Controller
{
    /**
     * Exibe a lista de produtos com o próximo código preparado
     */
    public function index()
    {
        $usuario = Auth::user();
        $proximoCodigo = $this->getNextCode();

        $query = Produto::query();

        if ($usuario->tipo_us !== 'ADMIN') {
            $query->where('nome_pro', 'LIKE', $usuario->acesso_us . '%');
        }

        $produtos = $query->orderBy('codigo_pro')->get();

        return view('produtos.index', compact('produtos', 'proximoCodigo'));
    }

    /**
     * API AJAX para obter o próximo código
     */
    public function nextCode()
    {
        return response()->json(['next' => $this->getNextCode()]);
    }

    /**
     * Salvar novo produto
     */
    public function store(Request $request)
    {
        $usuario = Auth::user();
        $usuarioCodigo = $usuario->codigo_us;

        $request->validate([
            'codigo_pro'   => 'unique:produtos',
            'nome_pro'     => 'required',
            'valor_pro'    => 'required|numeric',
            'valor_compra' => 'required|numeric',
            'quant_pro'    => 'required|integer|min:1',
            'data_pro'     => 'required|date'
        ]);

        $subs = $this->calcSubtotals($request->valor_pro, $request->valor_compra, $request->quant_pro);

        DB::transaction(function() use ($request, $subs, $usuarioCodigo) {
            // Criar produto
            Produto::create([
                'codigo_pro'     => $request->codigo_pro,
                'tipo_pro'       => $request->tipo_pro,
                'nome_pro'       => $request->nome_pro,
                'valor_pro'      => $request->valor_pro,
                'valor_compra'   => $request->valor_compra,
                'quant_pro'      => $request->quant_pro,
                'data_pro'       => $request->data_pro,
                'subtotalvenda'  => $subs['subtotalvenda'],
                'subtotalcompra' => $subs['subtotalcompra'],
                'subtotallucro'  => $subs['subtotallucro'],
            ]);

            // Registrar movimento
            DB::table('movimentos_estoque')->insert([
                'produto_codigo' => $request->codigo_pro,
                'usuario_codigo' => $usuarioCodigo,
                'tipo' => 'novo_produto',
                'quantidade' => $request->quant_pro,
                'custo_unitario' => $request->valor_compra,
                'referencia' => 'Criação de novo produto',
                'created_at' => now()
            ]);
        });

        return response()->json(['success' => true]);
    }

    /**
     * Form de edição
     */
    public function edit(Produto $produto)
    {
        return view('produtos.edit', compact('produto'));
    }

    /**
     * Atualizar produto
     */
  public function update(Request $request, $codigo_pro)
{
    $usuario = Auth::user();
    $usuarioCodigo = $usuario->codigo_us;

    $request->validate([
        'nome_pro'     => 'required',
        'valor_pro'    => 'required|numeric',
        'valor_compra' => 'required|numeric',
        'quant_pro'    => 'required|integer|min:0',
        'data_pro'     => 'required|date',
    ]);

    $produto = Produto::where('codigo_pro', $codigo_pro)->firstOrFail();

    // Calcula subtotais
    $subs = $this->calcSubtotals($request->valor_pro, $request->valor_compra, $request->quant_pro);

    DB::transaction(function() use ($produto, $request, $subs, $usuarioCodigo) {
        $quantAntes = $produto->quant_pro;
        $quantNova = $request->quant_pro;

        // Atualiza o produto
        $produto->update([
            'tipo_pro'       => $request->tipo_pro,
            'nome_pro'       => $request->nome_pro,
            'valor_pro'      => $request->valor_pro,
            'valor_compra'   => $request->valor_compra,
            'quant_pro'      => $quantNova,
            'data_pro'       => $request->data_pro,
            'subtotalvenda'  => $subs['subtotalvenda'],
            'subtotalcompra' => $subs['subtotalcompra'],
            'subtotallucro'  => $subs['subtotallucro'],
        ]);

        // 1️⃣ Registrar ajuste de estoque se houver diferença de quantidade
        $difQuant = $quantNova - $quantAntes;
        if ($difQuant != 0) {
            DB::table('movimentos_estoque')->insert([
                'produto_codigo' => $produto->codigo_pro,
                'usuario_codigo' => $usuarioCodigo,
                'tipo' => 'ajuste',
                'quantidade' => $difQuant,
                'custo_unitario' => $request->valor_compra,
                'referencia' => 'Ajuste de quantidade na atualização',
                'created_at' => now()
            ]);
        }

        // 2️⃣ Registrar movimento de atualização geral
        DB::table('movimentos_estoque')->insert([
            'produto_codigo' => $produto->codigo_pro,
            'usuario_codigo' => $usuarioCodigo,
            'tipo' => 'atualizacao',
            'quantidade' => $quantNova, // nunca nulo
            'custo_unitario' => $request->valor_compra,
            'referencia' => 'Atualização de dados do produto',
            'created_at' => now()
        ]);
    });

    return response()->json(['success' => true]);
}


    /**
     * Deletar produto
     */
    public function destroy($codigo)
    {
        $usuario = Auth::user();
        $usuarioCodigo = $usuario->codigo_us;

        $produto = Produto::where('codigo_pro', $codigo)->firstOrFail();

        DB::transaction(function() use ($produto, $usuarioCodigo) {
            // Registrar remoção
            DB::table('movimentos_estoque')->insert([
                'produto_codigo' => $produto->codigo_pro,
                'usuario_codigo' => $usuarioCodigo,
                'tipo' => 'remocao',
                'quantidade' => $produto->quant_pro,
                'custo_unitario' => $produto->valor_compra,
                'referencia' => 'Produto removido',
                'created_at' => now()
            ]);

            // Deletar produto
            $produto->delete();
        });

        return response()->json([
            'success' => true,
            'message' => 'Produto removido com sucesso!'
        ]);
    }

    /**
     * Gera o próximo código de produto (PRO0001...)
     */
    protected function getNextCode()
    {
        $ultimo = Produto::orderBy('codigo_pro', 'desc')->first();

        if ($ultimo && preg_match('/PRO(\d+)/', $ultimo->codigo_pro, $m)) {
            $num = intval($m[1]) + 1;
            return 'PRO' . str_pad($num, 4, '0', STR_PAD_LEFT);
        }

        return 'PRO0001';
    }

    /**
     * Calcula subtotais e lucro
     */
    private function calcSubtotals($valor, $valorCompra, $quant)
    {
        return [
            'subtotalvenda'  => $valor * $quant,
            'subtotalcompra' => $valorCompra * $quant,
            'subtotallucro'  => ($valor - $valorCompra) * $quant,
        ];
    }
}
