<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use PDF;

class EstoqueController extends Controller
{
    /**
     * Exibe o histórico de movimentações do estoque com filtros
     */
    public function historico(Request $request)
    {
        $query = DB::table('movimentos_estoque')
            ->leftJoin('produtos', 'movimentos_estoque.produto_codigo', '=', 'produtos.codigo_pro')
            ->leftJoin('usuarios', 'movimentos_estoque.usuario_codigo', '=', 'usuarios.codigo_us')
            ->select(
                'movimentos_estoque.*',
                'produtos.nome_pro',
                'produtos.codigo_pro as produto_codigo', // <-- adicionado
                'usuarios.nome_us as usuario_nome'
            );

        // Filtro por produto
        if ($request->produto) {
            $query->where(function ($q) use ($request) {
                $q->where('produtos.nome_pro', 'LIKE', "%{$request->produto}%")
                  ->orWhere('produtos.codigo_pro', 'LIKE', "%{$request->produto}%");
            });
        }

        // Filtro por datas
        if ($request->inicio) {
            $query->whereDate('movimentos_estoque.created_at', '>=', $request->inicio);
        }

        if ($request->fim) {
            $query->whereDate('movimentos_estoque.created_at', '<=', $request->fim);
        }

        // Escolha entre paginate (bom para muitas linhas) ou get() (para DataTables client-side)
        $movimentos = $query->orderBy('movimentos_estoque.created_at', 'DESC')
                            ->paginate(20); // alternativa: ->get() se preferir que DataTables faça paginação client-side

        return view('estoque.historico', compact('movimentos'));
    }

    /**
     * Exporta o histórico de movimentações em PDF com filtros aplicados
     */
    public function exportarPDF(Request $request)
    {
        $query = DB::table('movimentos_estoque')
            ->leftJoin('produtos', 'movimentos_estoque.produto_codigo', '=', 'produtos.codigo_pro')
            ->leftJoin('usuarios', 'movimentos_estoque.usuario_codigo', '=', 'usuarios.codigo_us')
            ->select(
                'movimentos_estoque.*',
                'produtos.nome_pro',
                'produtos.codigo_pro as produto_codigo', // manter consistência
                'usuarios.nome_us as usuario_nome'
            );

        // Reaplicar filtros da view
        if ($request->produto) {
            $query->where(function ($q) use ($request) {
                $q->where('produtos.nome_pro', 'LIKE', "%{$request->produto}%")
                  ->orWhere('produtos.codigo_pro', 'LIKE', "%{$request->produto}%");
            });
        }

        if ($request->inicio) {
            $query->whereDate('movimentos_estoque.created_at', '>=', $request->inicio);
        }

        if ($request->fim) {
            $query->whereDate('movimentos_estoque.created_at', '<=', $request->fim);
        }

        $movimentos = $query->orderBy('movimentos_estoque.created_at', 'DESC')
                            ->get(); // PDF não precisa de paginação

        $pdf = PDF::loadView('estoque.historico_pdf', compact('movimentos'))
                  ->setPaper('a4', 'portrait');

        return $pdf->download('historico-estoque.pdf');
    }
}
