<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use PDF;


class EntradaEstoqueController extends Controller
{
    /**
     * Listar entradas com filtros
     */
    public function index(Request $request)
    {
        $usuario = Auth::user();

        $produtoFiltro = $request->produto ?? '';
        $dataInicio = $request->data_inicio;
        $dataFim = $request->data_fim;

        $entradas = DB::table('entradas_estoque as e')
            ->join('produtos as p', 'p.codigo_pro', '=', 'e.produto_codigo')
            ->select(
                'e.id',
                'p.nome_pro',
                'e.quantidade',
                'e.preco_unitario',
                DB::raw('COALESCE(e.usuario, "Sistema") AS usuario'),
                'e.data'
            )
            ->when($usuario->tipo_us !== 'ADMIN', function($q) use ($usuario) {
                $q->where('p.nome_pro', 'LIKE', $usuario->acesso_us . '%');
            })
            ->when($produtoFiltro, function($q) use ($produtoFiltro) {
                $q->where('p.nome_pro', 'LIKE', "%{$produtoFiltro}%");
            })
            ->when($dataInicio, function($q) use ($dataInicio) {
                $q->whereDate('e.data', '>=', $dataInicio);
            })
            ->when($dataFim, function($q) use ($dataFim) {
                $q->whereDate('e.data', '<=', $dataFim);
            })
            ->orderBy('e.data', 'desc')
            ->get();

        if ($request->ajax()) {
            return view('estoque.partials.tabelaEntradas', compact('entradas'));
        }

        return view('estoque.index', compact('entradas'));
    }

    /**
     * Autocomplete produtos
     */
    public function buscarProduto(Request $request)
    {
        $term = trim($request->q);
        $usuario = Auth::user();

        if (!$term) return response()->json([]);

        $query = DB::table('produtos');
        if ($usuario->tipo_us !== 'ADMIN') {
            $query->where('nome_pro', 'LIKE', $usuario->acesso_us . '%');
        }

        $produtos = $query
            ->where(function($q) use ($term) {
                $q->where('nome_pro', 'LIKE', "%{$term}%")
                  ->orWhere('codigo_pro', 'LIKE', "%{$term}%");
            })
            ->limit(10)
            ->get();

        return response()->json(
            $produtos->map(fn($p) => [
                'id' => $p->codigo_pro,
                'text' => "{$p->nome_pro} (COD: {$p->codigo_pro}) — Estoque: {$p->quant_pro}",
                'preco' => $p->valor_compra
            ])
        );
    }

    /**
     * Registrar entrada de estoque
     */

public function store(Request $request)
{
    $request->validate([
        'produto_codigo' => 'required|string',
        'quantidade' => 'required|integer|min:1',
        'preco_unitario' => 'required|numeric|min:0'
    ]);

    $codigo = $request->produto_codigo;
    $quantidadeNova = $request->quantidade;
    $precoUnitario = $request->preco_unitario;

    $usuario = Auth::user();
    $usuarioNome = $usuario->nome_us ?? null;
    $usuarioCodigo = $usuario->codigo_us ?? null;

    // Buscar produto existente
    $produto = DB::table('produtos')->where('codigo_pro', $codigo)->first();
    if(!$produto){
        return response()->json(['error' => 'Produto não encontrado.'], 404);
    }

    // Atualizar quantidade e preço de compra
    $novaQuantidade = $produto->quant_pro + $quantidadeNova;
    DB::table('produtos')
        ->where('codigo_pro', $codigo)
        ->update([
            'quant_pro' => $novaQuantidade,
            'valor_compra' => $precoUnitario
        ]);

    // Inserir entrada na tabela entradas_estoque
    $entradaId = DB::table('entradas_estoque')->insertGetId([
        'produto_codigo' => $codigo,
        'quantidade' => $quantidadeNova,
        'preco_unitario' => $precoUnitario,
        'usuario' => $usuarioNome,
        'data' => now()
    ]);

    // **Registrar movimentação no estoque**
    DB::table('movimentos_estoque')->insert([
        'produto_codigo' => $codigo,
        'usuario_codigo' => $usuarioCodigo,
        'tipo' => 'entrada',
        'quantidade' => $quantidadeNova,
        'custo_unitario' => $precoUnitario,
        'referencia' => "Entrada ID #{$entradaId}",
        'created_at' => now()
    ]);

    return response()->json(['msg' => 'Entrada registrada com sucesso!']);
}

    /**
     * Deletar entrada de estoque
     */
    public function destroy($id)
    {
        $usuario = Auth::user();
        $usuarioCodigo = $usuario->codigo_us;

        $entrada = DB::table('entradas_estoque')->where('id', $id)->first();
        if (!$entrada) return response()->json(['error' => 'Entrada não encontrada'], 404);

        if ($usuario->tipo_us !== 'ADMIN') {
            $produto = DB::table('produtos')
                ->where('codigo_pro', $entrada->produto_codigo)
                ->where('nome_pro', 'LIKE', $usuario->acesso_us . '%')
                ->first();
            if (!$produto) return response()->json(['error' => 'Permissão negada'], 403);
        }

        $produto = DB::table('produtos')->where('codigo_pro', $entrada->produto_codigo)->first();
        if ($produto->quant_pro < $entrada->quantidade)
            return response()->json(['error' => 'Não é possível excluir, estoque insuficiente.'], 422);

        DB::transaction(function() use ($entrada, $usuarioCodigo, $id) {
            // Atualizar estoque
            DB::table('produtos')->where('codigo_pro', $entrada->produto_codigo)
                ->decrement('quant_pro', $entrada->quantidade);

            // Registrar movimento
            DB::table('movimentos_estoque')->insert([
                'produto_codigo' => $entrada->produto_codigo,
                'usuario_codigo' => $usuarioCodigo,
                'tipo' => 'exclusao_entrada',
                'quantidade' => -$entrada->quantidade,
                'custo_unitario' => $entrada->preco_unitario,
                'referencia' => "Exclusão Entrada ID #{$entrada->id}",
                'created_at' => now()
            ]);

            // Deletar entrada
            DB::table('entradas_estoque')->where('id', $id)->delete();
        });

        return response()->json(['msg' => 'Entrada removida e estoque ajustado.']);
    }

    public function exportarPDF(Request $request)
{
    $usuario = Auth::user();

    $query = DB::table('entradas_estoque as e')
        ->join('produtos as p', 'p.codigo_pro', '=', 'e.produto_codigo')
        ->select(
            'e.*',
            'p.nome_pro'
        );

    // Filtros da view
    if ($request->produto) {
        $query->where('p.nome_pro', 'LIKE', "%{$request->produto}%");
    }

    if ($request->data_inicio) {
        $query->whereDate('e.data', '>=', $request->data_inicio);
    }

    if ($request->data_fim) {
        $query->whereDate('e.data', '<=', $request->data_fim);
    }

    // Se não for admin, limitar acesso
    if ($usuario->tipo_us !== 'ADMIN') {
        $query->where('p.nome_pro', 'LIKE', $usuario->acesso_us . '%');
    }

    $entradas = $query->orderBy('e.data', 'desc')->get();

    $pdf = PDF::loadView('estoque.entradas_pdf', compact('entradas'))
              ->setPaper('a4', 'portrait');

    return $pdf->download('entradas-estoque.pdf');
}

}
