<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use PDF;

class DashboardController extends Controller
{

    private function getDados(Request $request)
    {
        $start = $request->start ? $request->start . " 00:00:00" : "1900-01-01 00:00:00";
        $end   = $request->end ? $request->end . " 23:59:59" : now();
        $categoriaFiltro = $request->categoria ?? 'todas';
        $produtoFiltro   = $request->produto !== "todos" ? $request->produto : null;
        $estoqueCritico  = $request->estoque_critico ?? 5;

        $produtosFiltro = DB::table('produtos')
            ->select('codigo_pro', 'nome_pro')
            ->orderBy('nome_pro')
            ->get();

        $produtos = DB::table('produtos')
            ->when($categoriaFiltro !== 'todas', fn($q) => $q->where('nome_pro', 'LIKE', $categoriaFiltro . '%'))
            ->when($produtoFiltro, fn($q) => $q->where('codigo_pro', $produtoFiltro))
            ->get();

        $getSetor = fn($nome) => strtoupper(substr(trim($nome), 0, 3));

        $totalEstoqueQtd   = $produtos->sum('quant_pro');
        $totalEstoqueValor = $produtos->sum(fn($p) => $p->quant_pro * $p->valor_compra);

        $barEstoque = $produtos->filter(fn($p) => $getSetor($p->nome_pro) === 'BAR');
        $armEstoque = $produtos->filter(fn($p) => $getSetor($p->nome_pro) === 'ARM');

        $barEstoqueQtd   = $barEstoque->sum('quant_pro');
        $barEstoqueValor = $barEstoque->sum(fn($p) => $p->quant_pro * $p->valor_compra);

        $armEstoqueQtd   = $armEstoque->sum('quant_pro');
        $armEstoqueValor = $armEstoque->sum(fn($p) => $p->quant_pro * $p->valor_compra);

        $criticosBar = $barEstoque->where('quant_pro', '<=', $estoqueCritico);
        $criticosArm = $armEstoque->where('quant_pro', '<=', $estoqueCritico);

        $vendas = DB::table('itensvenda as iv')
            ->join('vendas as v', 'v.codigo', '=', 'iv.codigoVenda')
            ->join('produtos as p', 'p.codigo_pro', '=', 'iv.codProduto')
            ->select('iv.quantidade', 'iv.subtotal', 'p.valor_compra', 'p.nome_pro')
            ->whereBetween('v.data', [$start, $end])
            ->when($categoriaFiltro !== 'todas', fn($q) => $q->where('p.nome_pro', 'LIKE', $categoriaFiltro . '%'))
            ->when($produtoFiltro, fn($q) => $q->where('p.codigo_pro', $produtoFiltro))
            ->get();

        $totalVendas = $vendas->sum('subtotal');
        $totalLucro  = $vendas->sum(fn($v) => $v->subtotal - ($v->quantidade * $v->valor_compra));

        $barTotalVendido = $vendas->filter(fn($v) => $getSetor($v->nome_pro) === 'BAR')->sum('subtotal');
        $armTotalVendido = $vendas->filter(fn($v) => $getSetor($v->nome_pro) === 'ARM')->sum('subtotal');

        $barLucro = $vendas->filter(fn($v) => $getSetor($v->nome_pro) === 'BAR')
            ->sum(fn($i) => $i->subtotal - ($i->quantidade * $i->valor_compra));

        $armLucro = $vendas->filter(fn($v) => $getSetor($v->nome_pro) === 'ARM')
            ->sum(fn($i) => $i->subtotal - ($i->quantidade * $i->valor_compra));

        $vendasMensais = DB::table('itensvenda as iv')
            ->join('vendas as v', 'iv.codigoVenda', '=', 'v.codigo')
            ->selectRaw('DATE_FORMAT(v.data, "%m/%Y") as mes, SUM(iv.subtotal) as total')
            ->groupBy('mes')
            ->orderBy('v.data')
            ->get();

        $lucroMensal = DB::table('itensvenda as iv')
            ->join('vendas as v','v.codigo','=','iv.codigoVenda')
            ->join('produtos as p','p.codigo_pro','=','iv.codProduto')
            ->selectRaw('DATE_FORMAT(v.data,"%m/%Y") as mes, SUM(iv.subtotal - (iv.quantidade * p.valor_compra)) as lucro')
            ->groupBy('mes')
            ->orderBy('v.data')
            ->get();

        $maisVendidos = DB::table('itensvenda as iv')
            ->join('produtos as p', 'iv.codProduto', '=', 'p.codigo_pro')
            ->select('p.nome_pro', DB::raw('SUM(iv.quantidade) as quant'))
            ->groupBy('p.nome_pro')
            ->orderByDesc('quant')
            ->limit(10)
            ->get();

        return compact(
            'produtosFiltro','criticosBar','criticosArm',
            'barTotalVendido','barLucro','barEstoqueQtd','barEstoqueValor',
            'armTotalVendido','armLucro','armEstoqueQtd','armEstoqueValor',
            'totalVendas','totalLucro','totalEstoqueQtd','totalEstoqueValor',
            'vendasMensais','lucroMensal','maisVendidos'
        );
    }


    public function index(Request $request)
    {
        $dados = $this->getDados($request);
        return view('dashboard', $dados);
    }


    public function exportPdf(Request $request)
    {
        $dados = $this->getDados($request);

        $pdf = PDF::loadView('dashboard.report_pdf', $dados);

        return $pdf->stream("Relatorio-Dashboard.pdf");
    }
}
