<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\Usuario;

class AuthController extends Controller
{
    /**
     * Mostra a página de login.
     */
    public function loginPage()
    {
        return view('login');
    }

    /**
     * Faz o login do usuário.
     */
    public function login(Request $request)
    {
        $request->validate([
            'usuario' => 'required',
            'senha'   => 'required',
        ]);

        $user = Usuario::where('nome_us', $request->usuario)->first();

        if (!$user) {
            return back()->with('fail', 'Credenciais inválidas.');
        }

        $plain = (string) $request->senha;
        $stored = (string) $user->senha;

        if ($this->checkPassword($plain, $stored)) {
            // Re-hash se necessário
            if (!preg_match('/^\$2y\$|^\$2a\$|^\$argon2/i', $stored) || Hash::needsRehash($stored)) {
                $user->senha = Hash::make($plain);
                $user->save();
            }

            // Login
            Auth::login($user);
            $request->session()->regenerate();

            // Redirecionamento baseado no tipo_us
            if ($user->tipo_us === 'ADMIN') {
                return redirect()->intended(route('dashboard')); // admin
            } else {
                return redirect()->intended(route('caixa.index')); // usuário normal
            }
        }

        return back()->with('fail', 'Credenciais inválidas.');
    }

    /**
     * Logout do usuário.
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('login');
    }

    /**
     * Verifica a senha do usuário considerando bcrypt, argon, md5 ou texto plano.
     */
    private function checkPassword(string $plain, string $stored): bool
    {
        // bcrypt/argon
        if (preg_match('/^\$2y\$|^\$2a\$|^\$argon2/i', $stored)) {
            return Hash::check($plain, $stored);
        }

        // MD5
        if (preg_match('/^[0-9a-f]{32}$/i', $stored)) {
            return md5($plain) === $stored;
        }

        // Texto plano
        return $plain === $stored;
    }
}
